<?php
// If this file is called directly, abort.
if (!defined('WPINC')) {
    die;
}
/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://text.lk
 * @since      1.0.0
 *
 * @package    TextLK
 * @subpackage TextLK/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    TextLK
 * @subpackage TextLK/admin
 * @author     TextLK Developer Team <support@text.lk>
 */
class TextLK_Admin
{

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string $plugin_name The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string $version The current version of this plugin.
     */
    private $version;

    /**
     * Initialize the class and set its properties.
     *
     * @param string $plugin_name The name of this plugin.
     * @param string $version The version of this plugin.
     * @since    1.0.0
     */
    public function __construct($plugin_name, $version)
    {

        $this->plugin_name = $plugin_name;
        $this->version = $version;
    }

    /**
     * Register the stylesheets for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_styles()
    {

        /**
         * This function is provided for demonstration purposes only.
         *
         * An instance of this class should be passed to the run() function
         * defined in TextLK_Loader as all of the hooks are defined
         * in that particular class.
         *
         * The TextLK_Loader will then create the relationship
         * between the defined hooks and the functions defined in this
         * class.
         */

        wp_enqueue_style($this->plugin_name, plugin_dir_url(__FILE__) . 'css/textlk-admin.css', [], $this->version,
            'all');
    }

    /**
     * Register the JavaScript for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_scripts()
    {

        /**
         * This function is provided for demonstration purposes only.
         *
         * An instance of this class should be passed to the run() function
         * defined in TextLK_Loader as all of the hooks are defined
         * in that particular class.
         *
         * The TextLK_Loader will then create the relationship
         * between the defined hooks and the functions defined in this
         * class.
         */

        wp_enqueue_script($this->plugin_name, plugin_dir_url(__FILE__) . 'js/textlk-admin.js', ['jquery'],
            $this->version, false);
    }

    /**
     * Render the settings page for this plugin.
     *
     * @since    1.0.0
     */
    public function display_setting_page()
    {
        require_once 'partials/' . $this->plugin_name . '-admin-display_settings.php';
    }

    /**
     * Render the campaign page for this plugin.
     *
     * @since    1.0.0
     */
    public function display_campaign_page()
    {
        require_once 'partials/' . $this->plugin_name . '-admin-display_campaign.php';
    }

    /**
     *  Add the main menu and sub menu of the plugin
     *
     * @since    1.0.0
     */

    public function add_admin_menu()
    {
        $icon_url = plugins_url('../assets/textlk_icon.png', __FILE__);

        // Primary Main menu
        add_menu_page(
            'Text.lk SMS',
            'Text.lk SMS',
            'manage_options',
            $this->plugin_name,
            [$this, 'display_campaign_page'],
            $icon_url,
            76
        );

        add_submenu_page($this->plugin_name, 'SMS Campaign', 'Campaign', 'manage_options', $this->plugin_name,
            [$this, 'display_campaign_page']);

        add_submenu_page($this->plugin_name, 'Text.lk SMS Settings', 'Settings', 'manage_options',
            $this->plugin_name . '_settings', [$this, 'display_setting_page']);
    }

    /**
     * Add settings action link to the plugins page.
     *
     * @since    1.0.0
     */
    public function add_action_links($links)
    {

        /**
         * Documentation : https://codex.wordpress.org/Plugin_API/Filter_Reference/plugin_action_links_(plugin_file_name)
         * The "plugins.php" must match with the previously added add_submenu_page first option.
         * For custom post type you have to change 'plugins.php?page=' to 'edit.php?post_type=your_custom_post_type&page='
         */
        $settings_link = [
            '<a href="' . admin_url('admin.php?page=' . $this->plugin_name . '_settings') . '">' . __('Settings', 'textlk-wp') . '</a>',
        ];

        // -- OR --

        // $settings_link = array( '<a href="' . admin_url( 'options-general.php?page=' . $this->plugin_name ) . '">' . __( 'Settings', $this->plugin_name ) . '</a>', );

        return array_merge($settings_link, $links);
    }

    /**
     * Validate fields from admin area plugin settings form ('exopite-lazy-load-xt-admin-display.php')
     * @param mixed $input as field form settings form
     * @return mixed as validated fields
     */
    public function validate($input)
    {
        $options = get_option($this->plugin_name);

        if (!empty($input['textlk_sms_api_key']) && strpos(esc_attr($input['textlk_sms_api_key']), str_repeat('*', 24), '12')) {

            $input['textlk_sms_api_key'] = $options['textlk_sms_api_key'];

        }

        $options['textlk_sms_api_key'] = (isset($input['textlk_sms_api_key']) && !empty($input['textlk_sms_api_key'])) ? esc_attr($input['textlk_sms_api_key']) : '';
        $options['sender_id'] = (isset($input['sender_id']) && !empty($input['sender_id'])) ? esc_attr($input['sender_id']) : '';

        $options['order_status'] = (isset($input['order_status']) && !empty($input['order_status'])) ? 1 : 0;
        $options['wp_reg'] = (isset($input['wp_reg']) && !empty($input['wp_reg'])) ? 1 : 0;
        $options['wp_login'] = (isset($input['wp_login']) && !empty($input['wp_login'])) ? 1 : 0;
        $options['wc_reg'] = (isset($input['wc_reg']) && !empty($input['wc_reg'])) ? 1 : 0;
        $options['wc_login'] = (isset($input['wc_login']) && !empty($input['wc_login'])) ? 1 : 0;
        $options['otp_checkout'] = (isset($input['otp_checkout']) && !empty($input['otp_checkout'])) ? 1 : 0;
        $options['admin_phones'] = (isset($input['admin_phones']) && !empty($input['admin_phones'])) ? esc_attr($input['admin_phones']) : '';

        $options['order_status_pending'] = (isset($input['order_status_pending']) && !empty($input['order_status_pending'])) ? 1 : 0;
        $options['order_status_processing'] = (isset($input['order_status_processing']) && !empty($input['order_status_processing'])) ? 1 : 0;
        $options['order_status_on_hold'] = (isset($input['order_status_on_hold']) && !empty($input['order_status_on_hold'])) ? 1 : 0;
        $options['order_status_completed'] = (isset($input['order_status_completed']) && !empty($input['order_status_completed'])) ? 1 : 0;
        $options['order_status_cancelled'] = (isset($input['order_status_cancelled']) && !empty($input['order_status_cancelled'])) ? 1 : 0;
        $options['order_status_refunded'] = (isset($input['order_status_refunded']) && !empty($input['order_status_refunded'])) ? 1 : 0;
        $options['order_status_failed'] = (isset($input['order_status_failed']) && !empty($input['order_status_failed'])) ? 1 : 0;
        $options['order_status_admin'] = (isset($input['order_status_admin']) && !empty($input['order_status_admin'])) ? 1 : 0;

        $options['order_status_receipt_approval'] = (isset($input['order_status_receipt_approval']) && !empty($input['order_status_receipt_approval'])) ? 1 : 0;
        $options['order_status_receipt_upload'] = (isset($input['order_status_receipt_upload']) && !empty($input['order_status_receipt_upload'])) ? 1 : 0;
        $options['order_status_shipped'] = (isset($input['order_status_shipped']) && !empty($input['order_status_shipped'])) ? 1 : 0;
        $options['order_status_packed'] = (isset($input['order_status_packed']) && !empty($input['order_status_packed'])) ? 1 : 0;

        $options['ORDER_STATUS_PENDING_SMS'] = (isset($input['ORDER_STATUS_PENDING_SMS']) && !empty($input['ORDER_STATUS_PENDING_SMS'])) ? esc_attr($input['ORDER_STATUS_PENDING_SMS']) : '';
        $options['ORDER_STATUS_PROCESSING_SMS'] = (isset($input['ORDER_STATUS_PROCESSING_SMS']) && !empty($input['ORDER_STATUS_PROCESSING_SMS'])) ? esc_attr($input['ORDER_STATUS_PROCESSING_SMS']) : '';
        $options['ORDER_STATUS_ON_HOLD_SMS'] = (isset($input['ORDER_STATUS_ON_HOLD_SMS']) && !empty($input['ORDER_STATUS_ON_HOLD_SMS'])) ? esc_attr($input['ORDER_STATUS_ON_HOLD_SMS']) : '';
        $options['ORDER_STATUS_COMPLETED_SMS'] = (isset($input['ORDER_STATUS_COMPLETED_SMS']) && !empty($input['ORDER_STATUS_COMPLETED_SMS'])) ? esc_attr($input['ORDER_STATUS_COMPLETED_SMS']) : '';
        $options['ORDER_STATUS_CANCELLED_SMS'] = (isset($input['ORDER_STATUS_CANCELLED_SMS']) && !empty($input['ORDER_STATUS_CANCELLED_SMS'])) ? esc_attr($input['ORDER_STATUS_CANCELLED_SMS']) : '';
        $options['ORDER_STATUS_REFUNDED_SMS'] = (isset($input['ORDER_STATUS_REFUNDED_SMS']) && !empty($input['ORDER_STATUS_REFUNDED_SMS'])) ? esc_attr($input['ORDER_STATUS_REFUNDED_SMS']) : '';
        $options['ORDER_STATUS_FAILED_SMS'] = (isset($input['ORDER_STATUS_FAILED_SMS']) && !empty($input['ORDER_STATUS_FAILED_SMS'])) ? esc_attr($input['ORDER_STATUS_FAILED_SMS']) : '';

        $options['ORDER_STATUS_RECEIPT_APPROVAL_SMS'] = (isset($input['ORDER_STATUS_RECEIPT_APPROVAL_SMS']) && !empty($input['ORDER_STATUS_RECEIPT_APPROVAL_SMS'])) ? esc_attr($input['ORDER_STATUS_RECEIPT_APPROVAL_SMS']) : '';
        $options['ORDER_STATUS_RECEIPT_UPLOAD_SMS'] = (isset($input['ORDER_STATUS_RECEIPT_UPLOAD_SMS']) && !empty($input['ORDER_STATUS_RECEIPT_UPLOAD_SMS'])) ? esc_attr($input['ORDER_STATUS_RECEIPT_UPLOAD_SMS']) : '';
        $options['ORDER_STATUS_SHIPPED_SMS'] = (isset($input['ORDER_STATUS_SHIPPED_SMS']) && !empty($input['ORDER_STATUS_SHIPPED_SMS'])) ? esc_attr($input['ORDER_STATUS_SHIPPED_SMS']) : '';
        $options['ORDER_STATUS_PACKED_SMS'] = (isset($input['ORDER_STATUS_PACKED_SMS']) && !empty($input['ORDER_STATUS_PACKED_SMS'])) ? esc_attr($input['ORDER_STATUS_PACKED_SMS']) : '';
        
        $options['ADMIN_STATUS_SMS'] = (isset($input['ADMIN_STATUS_SMS']) && !empty($input['ADMIN_STATUS_SMS'])) ? esc_attr($input['ADMIN_STATUS_SMS']) : '';
       
        if (!$this->checkAPI($options['textlk_sms_api_key'])) {

            $options['order_status'] =
            $options['wp_reg'] =
            $options['wp_login'] =
            $options['wc_reg'] =
            $options['wc_login'] =
            $options['otp_checkout'] =
            $options['order_status_pending'] =
            $options['order_status_processing'] =
            $options['order_status_on_hold'] =
            $options['order_status_completed'] =
            $options['order_status_cancelled'] =
            $options['order_status_refunded'] =
            $options['order_status_failed'] =
            $options['order_status_admin'] = 0;

            add_settings_error(
                $this->plugin_name, // Slug title of setting
                $this->plugin_name, // Slug-name , Used as part of 'id' attribute in HTML output.
                __('Please configure a valid SMS API Key.', 'textlk-wp'),
                // message text, will be shown inside styled <div> and <p> tags
                'error' // Message type, controls HTML class. Accepts 'error' or 'updated'.
            );
        }

        return $options;
    }

    /**
     * Check if entered api key is valid or not
     * @return bool
     */
    private function checkAPI($textlk_sms_api_key)
    {
        if (empty($textlk_sms_api_key)) {
            return false;
        }

        require_once TEXTLK_WP_PATH . 'includes/sms.class.php';

        $smsPortal = new TextLK_SMS($textlk_sms_api_key);

        $response = $smsPortal->getBalance();

        return $response && $response->error === 0;
    }

    /**
     * update all settings
     */
    public function options_update()
    {
        register_setting($this->plugin_name, $this->plugin_name, [
            'sanitize_callback' => [$this, 'validate'],
        ]);
    }

    /**
     * send campaign msg to users
     */
    public function textlk_sms_send_campaign()
    {
        $numbersArr = [];

        $numbers = (isset($_POST[$this->plugin_name]['numbers']) && !empty($_POST[$this->plugin_name]['numbers'])) ? sanitize_textarea_field($_POST[$this->plugin_name]['numbers']) : '';
        $include_all_users = (isset($_POST[$this->plugin_name]['all_users']) && !empty($_POST[$this->plugin_name]['all_users'])) ? 1 : 0;
        $body = (isset($_POST[$this->plugin_name]['body']) && !empty($_POST[$this->plugin_name]['body'])) ? sanitize_textarea_field($_POST[$this->plugin_name]['body']) : false;

        //Grab all options
        $options = get_option($this->plugin_name);
        $textlk_sms_api_key = !empty($options['textlk_sms_api_key']) ? $options['textlk_sms_api_key'] : '';
        $sender_id = !empty($options['sender_id']) ? trim($options['sender_id']) : '';

        // Empty body
        if (!$body) {
            $this->add_flash_notice(__("Fill the required fields properly", "textlk-wp"), "error");

            // Redirect to plugin page
            wp_redirect($_SERVER['HTTP_REFERER']);
            exit();
        }
        if (!$textlk_sms_api_key) {
            $this->add_flash_notice(__("No valid API Key is set.", "textlk-wp"), "error");

            // Redirect to plugin page
            wp_redirect($_SERVER['HTTP_REFERER']);
            exit();
        }

        if ($numbers) {
            // split by new line
            $numbersArr = explode(PHP_EOL, $numbers);
        }

        if ($include_all_users) {
            $woo_numbers = $this->getCustomersPhone();
            $numbersArr = array_merge($numbersArr, $woo_numbers);
        }

        // Final Numbers
        $numbers = implode(',', $numbersArr);

        require_once TEXTLK_WP_PATH . 'includes/sms.class.php';

        $sms = new TextLK_SMS($textlk_sms_api_key);
        $sms->numbers = $numbers;
        $sms->body = $body;
        $sms->sender_id = $sender_id;

        $response = $sms->Send();

        if (!$response) {
            $this->add_flash_notice(__("Something went wrong, please try again.", "textlk-wp"), "error");

        } elseif ($response->error !== 0) {
            // Fallback error message if no error message is provided by the API
            $error_msg = !empty($response->msg) ? $response->msg : __('An unknown error occurred while processing your request.', 'textlk-wp');
            $this->add_flash_notice($error_msg, 'error');
        } else {
            // Fallback success message if no success message is provided by the API
            $success_msg = !empty($response->msg) ? $response->msg : __('Your request was processed successfully.', 'textlk-wp');
            $this->add_flash_notice($success_msg, 'success');
        }

        // Redirect to plugin page
        wp_redirect($_SERVER['HTTP_REFERER']);
        exit();
    }

    /**
     * Add a flash notice to {prefix}options table until a full page refresh is done
     *
     * @param string $notice our notice message
     * @param string $type This can be "info", "warning", "error" or "success", "warning" as default
     * @param boolean $dismissible set this to TRUE to add is-dismissible functionality to your notice
     * @return void
     */

    public function add_flash_notice($notice = "", $type = "warning", $dismissible = true)
    {
        // Here we return the notices saved on our option, if there are no notices, then an empty array is returned
        $notices = get_option($this->plugin_name . '_notices', []);

        $dismissible_text = ($dismissible) ? "is-dismissible" : "";

        // We add our new notice.
        $notices[] = [
            "notice" => $notice,
            "type" => $type,
            "dismissible" => $dismissible_text,
        ];

        // Then we update the option with our notices array
        update_option('textlk_notices', $notices);
    }

    public function getCustomersPhone()
	{
		// Get all users who have a non-empty 'billing_phone' and are not administrators
		$args = [
			'meta_query' => [
				[
					'key'     => 'billing_phone',
					'value'   => '',
					'compare' => '!=', // Ensure phone number is not empty
				]
			],
			'role__not_in' => ['administrator'], // Exclude administrators
			'fields' => ['ID'], // Return only user IDs to minimize the data
		];

		// Fetch users based on the arguments
		$users = get_users($args);

		// Extract phone numbers from user meta
		$phones = [];
		foreach ($users as $user) {
			$phone = get_user_meta($user->ID, 'billing_phone', true);
			if (!empty($phone)) {
				$phones[] = $phone;
			}
		}

		return array_unique($phones); // Remove duplicates
	}

    /**
     * Function executed when the 'admin_notices' action is called, here we check if there are notices on
     * our database and display them, after that, we remove the option to prevent notices being displayed forever.
     * @return void
     */

    public function display_flash_notices()
    {
        $notices = get_option($this->plugin_name . '_notices', []);

        // Iterate through our notices to be displayed and print them.
        foreach ($notices as $notice) {
			printf(
				'<div class="notice notice-%1$s %2$s"><p>%3$s</p></div>',
				esc_attr($notice['type']),
				esc_attr($notice['dismissible']),
				esc_html($notice['notice'])
			);
		}

        // Now we reset our options to prevent notices being displayed forever.
        if (!empty($notices)) {
            delete_option($this->plugin_name . '_notices', []);
        }
    }

    public function add_admin_body_class($classes)
    {
        // Check if the current admin page belongs to your plugin
        $screen = get_current_screen();
        if (strpos($screen->id, $this->plugin_name) !== false) {
            // Add your custom classes here
            $classes .= 'textlk-admin-page';
        }
        return $classes;
    }

}
