<?php

// If this file is called directly, abort.
if (!defined('WPINC')) {
	die;
}

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the public-facing stylesheet and JavaScript.
 *
 * @package    TextLK
 * @subpackage TextLK/public
 * @author     TextLK Developer Team <support@text.lk>
 */
class TextLK_Public
{

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string $plugin_name The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string $version The current version of this plugin.
	 */
	private $version;
	private $options;
	/**
	 * @var false
	 */
	private $pluginActive;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @param  string  $plugin_name  The name of the plugin.
	 * @param  string  $version      The version of this plugin.
	 *
	 * @since    1.0.0
	 */
	public function __construct($plugin_name, $version)
	{
		$this->plugin_name  = $plugin_name;
		$this->version      = $version;
		$this->options      = get_option($this->plugin_name);
		$this->pluginActive = !empty($this->options['textlk_sms_api_key']);
	}

	/**
	 * @return void
	 * @since 1.0.0
	 * start session if not started
	 */
	public function start_session_wp()
	{
		if (session_status() === PHP_SESSION_NONE) {
			session_start();
		}
	}

	/**
	 * Register the stylesheets for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles()
	{
		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in TextLK_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The TextLK_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_style(
			$this->plugin_name,
			plugin_dir_url(__FILE__) . 'css/textlk-public.css',
			[],
			$this->version,
			'all'
		);
	}

	/**
	 * Register the JavaScript for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts()
	{
		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in TextLK_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The TextLK_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_script(
			$this->plugin_name,
			plugin_dir_url(__FILE__) . 'js/textlk-public.js',
			['jquery'],
			$this->version,
			false
		);

		// adding a js variable for ajax form submit url
		wp_localize_script(
			$this->plugin_name,
			$this->plugin_name . '_object',
			['ajaxurl' => admin_url('admin-ajax.php')]
		);
	}

	/**
	 * Woocommerce
	 * show phone number on register page and my account
	 */
	public function wc_phone_on_register()
	{
		if (!$this->pluginActive || !$this->options['wc_reg']) {
			return;
		}

		$user  = wp_get_current_user();
		$value = isset($_POST['billing_phone']) ? sanitize_text_field($_POST['billing_phone'])
			: $user->billing_phone;
?>

		<p class="woocommerce-form-row woocommerce-form-row--wide form-row form-row-wide">
			<label for="reg_billing_phone"><?php esc_html_e('Phone', 'textlk-wp'); ?> <span class="required">*</span>
			</label>
			<input type="tel" minlength="9" maxlength="11" class="input-text" name="billing_phone" id="reg_billing_phone" value="<?php echo esc_attr($value) ?>" required />
		</p>
		<div class="clear"></div>

	<?php
	}

	/**
	 *  Default WordPress
	 * show otp form in registration form
	 */
	public function add_otp_field_on_wp_reg_form()
	{
		if (!$this->pluginActive || !$this->options['wp_reg']) {
			return;
		}
		require_once 'partials/add-otp-on-login-form.php';
	?>
		<input type='hidden' name='action_type' id='action_type' value='wp_reg' />
	<?php
	}

	/**
	 *  Woocommerce
	 * show otp form in registration form
	 */
	public function add_otp_field_on_wc_reg_form()
	{
		if (!$this->pluginActive || !$this->options['wc_reg']) {
			return;
		}

		require_once 'partials/add-otp-on-wc-reg-form.php';
	?>
		<input type='hidden' name='action_type' id='action_type' value='wc_reg' />
	<?php
	}

	/**
	 * Woocommerce + Default WordPress
	 * ajax otp send on post phone number *
	 */
	public function send_otp_for_reg()
	{
		$user_phone = $user_email = '';

		if (isset($_POST['billing_phone'], $_POST['email'])) {
			$user_phone = $this->validateNumber(sanitize_text_field($_POST['billing_phone'])); 
			$user_email = sanitize_text_field($_POST['email']);
		}

		if (!$user_email && !empty($_POST['billing_email'])) {
			$user_email = sanitize_text_field($_POST['billing_email']);
		}

		if (!filter_var($user_email, FILTER_VALIDATE_EMAIL)) {
			$response = ['status' => 400, 'message' => __('The email address you entered is not valid!', 'textlk-wp')];
			echo wp_kses_post(json_encode($response));
			wp_die();
			exit;
		}

		if (isset($_POST['password']) && empty($_POST['password']) && strlen($_POST['password']) < 8) {
			$response = ['status' => 400, 'message' => __('Weak - Please enter a stronger password.', 'textlk-wp')];
			echo wp_kses_post(json_encode($response));
			wp_die();
			exit;
		}

		if (!$user_phone) {
			$response = ['status' => 400, 'message' => __('The phone number you entered is not valid!', 'textlk-wp')];
			echo wp_kses_post(json_encode($response));
			wp_die();
			exit;
		}

		//we will send sms
		$otp_code = $this->generateOTP();

		$body = 'Your OTP for ' . get_bloginfo() . ' registration is: ' . $otp_code . '. Valid for 2 minutes.';

		if (!empty($_POST['action_type']) && $_POST['action_type'] === 'wc_checkout') {
			$body = 'Your OTP for secure order checkout on ' . get_bloginfo() . ' is: ' . $otp_code . '. Use it within 2 min to complete the checkout process.';
		}

		$sms_response = $this->SendSMS($user_phone, $body);

		if ($sms_response->error === 0) {
			// save info in database for later verification
			if ($this->log_login_register_action(
				$user_phone,
				$otp_code
			)) {
				$response = [
					'status'  => 200,
					'message' => 'A OTP (One Time Passcode) has been sent. Please enter the OTP in the field below to verify your phone.',
				];
			} else {
				$response = ['status' => 400, 'message' => __('Error occurred while sending OTP. Please try again.', 'textlk-wp')];
			}

			echo wp_kses_post(json_encode($response));
			wp_die();
			exit;
		}

		$response = ['status' => '400', 'message' => __('Error occurred while sending OTP. Contact Administrator.', 'textlk-wp')];
		echo wp_kses_post(json_encode($response));
		wp_die();
		exit;
	}

	public function reformatPhoneNumbers($value)
	{
		// Remove non-numeric characters
		$number = preg_replace("/[^0-9]/", "", $value);
		
		// Handle different lengths and prefixes
		if (strlen($number) == 9) {
			// Add country code for 9-digit numbers
			$number = "94" . $number;
		} elseif (strlen($number) == 10 && substr($number, 0, 1) == '0') {
			// Remove leading 0 and add country code for 10-digit numbers
			$number = "94" . substr($number, 1);
		} elseif (strlen($number) == 12 && substr($number, 0, 3) == '940') {
			// Remove '940' prefix and add '94' for 12-digit numbers
			$number = "94" . substr($number, 3);
		}
		
		return $number;
	}

	/**
	 * Validate Sri Lanka phone number format
	 *
	 * @param $num
	 *
	 * @return false|int|string
	 */
	public function validateNumber($num)
	{
		if (!$num) {
			return false;
		}

		// Reformat the number (remove spaces, dashes, etc.)
		$num = $this->reformatPhoneNumbers($num);

		// Normalize the number: Remove leading + and 00
		$num = ltrim($num, "+");
		$num = preg_replace('/^00/', '', $num);

		// If it doesn't start with 94, assume it's a local number (e.g., 0701234567)
		if (substr($num, 0, 2) !== '94') {
			// Remove leading zero and prepend 94
			$num = '94' . ltrim($num, '0');
		}

		// Valid mobile prefixes in Sri Lanka
		$validPrefixes = ["9470", "9471", "9472", "9474", "9475", "9476", "9477", "9478"];

		// Validate the number
		if (
			is_numeric($num) && 
			strlen($num) === 11 && 
			in_array(substr($num, 0, 4), $validPrefixes, true)
		) {
			return $num;
		}

		return false;
	}

	/**
	 * Generate 6 digit otp code
	 *
	 * @return string
	 */
	public function generateOTP()
	{
		$otp = '';

		for ($i = 0; $i < 6; $i++) {
			$otp .= wp_rand(0, 9);
		}

		return $otp;
	}

	/**
	 * Send SMS via sms api
	 *
	 * @param $to
	 * @param $body
	 *
	 * @return false|mixed
	 */
	public function SendSMS($to, $body)
	{
		if (!$this->pluginActive) {
			return false;
		}

		$textlk_sms_api_key   = !empty($this->options['textlk_sms_api_key']) ? $this->options['textlk_sms_api_key'] : '';
		$sender_id = !empty($this->options['sender_id']) ? trim($this->options['sender_id']) : '';

		require_once TEXTLK_WP_PATH . 'includes/sms.class.php';

		$sms            = new TextLK_SMS($textlk_sms_api_key);
		$sms->numbers   = $to;
		$sms->body      = $body;
		$sms->sender_id = $sender_id;

		return $sms->Send();
	}

	/**
	 * after sending otp to user, log the otp and data in db
	 *
	 * @param $mobile_phone
	 * @param $otp_code
	 *
	 * @return bool
	 */
	public function log_login_register_action(
		$mobile_phone,
		$otp_code
	) {
		$dateTime = new DateTime(TEXTLK_WP_TIMESTAMP);
		$dateTime->modify('+2 minutes');

		$_SESSION['textlk_otp_code'] = $otp_code;
		$_SESSION['textlk_expires']  = $dateTime->format('Y-m-d H:i:s');

		if (!empty($_SESSION['textlk_otp_code'])) {
			return true;
		}

		return false;
	}

	/**
	 * Verify otp and register the user
	 *
	 * @param $customer_id
	 */
	public function register_the_customer($customer_id)
	{
		if (!$this->pluginActive || (!$this->options['wp_reg'] && !$this->options['wc_reg'])) {
			return;
		}
		if (isset($_POST['billing_phone']) && $this->validateNumber(sanitize_text_field($_POST['billing_phone']))) {
			update_user_meta(
				$customer_id,
				'billing_phone',
				sanitize_text_field($this->validateNumber($_POST['billing_phone']))
			);
		}
	}

	/**
	 * Default WordPress
	 * show phone number on register page
	 */
	public function wp_phone_on_register()
	{
		if (!$this->pluginActive || !$this->options['wp_reg']) {
			return;
		}

		$billing_phone = (!empty($_POST['billing_phone'])) ? sanitize_text_field($_POST['billing_phone']) : '';

	?>
		<p>
			<label for="billing_phone"><?php esc_html_e('Phone', 'textlk-wp'); ?><br />
				<input type="text" name="billing_phone" id="reg_billing_phone" class="input" value="<?php echo esc_attr($billing_phone); ?>" size="25" /></label>
		</p>
	<?php
	}

	/**
	 * WordPress validate phone and validate otp
	 *
	 * @param $errors
	 * @param $sanitized_user_login
	 * @param $user_email
	 *
	 * @return mixed
	 */
	public function wp_register_form_validation($errors, $sanitized_user_login, $user_email)
	{
		if (
			$this->pluginActive && $this->options['wp_reg'] && !empty($_POST['action_type']) &&
			$_POST['action_type'] === 'wp_reg'
		) {
			$this->register_form_validation($errors, $sanitized_user_login, $user_email);
		}

		return $errors;
	}

	/**
	 * Register Form Validation
	 *
	 * @param $errors
	 * @param $sanitized_user_login
	 * @param $user_email
	 *
	 * @return mixed
	 */
	public function register_form_validation($errors, $sanitized_user_login, $user_email)
	{

		$enable_guest_checkout = get_option('woocommerce_enable_guest_checkout');
		$enable_guest_checkout = $enable_guest_checkout === 'yes' ? true : false;

		if (!$this->pluginActive) {
			return $errors;
		}

		if (
			empty($_REQUEST['billing_phone']) || !is_numeric($_REQUEST['billing_phone']) ||
			!$this->validateNumber(sanitize_text_field($_REQUEST['billing_phone']))
		) {
			$errors->add('phone_error', __('You phone number is not valid.', 'textlk-wp'));
		}

		$billing_phone = $this->validateNumber(sanitize_text_field($_REQUEST['billing_phone']));

		$hasPhoneNumber = get_users('meta_value=' . $billing_phone);

		if (!empty($hasPhoneNumber)) {
			$errors->add('duplicate_phone_error', __('Mobile number is already used!', 'textlk-wp'));
		}

		if (!empty($_REQUEST['otp_code'])) {
			$otp_code = sanitize_text_field($_REQUEST['otp_code']);

			$valid_user = $this->authenticate_otp(trim($otp_code));

			if ($valid_user) {
				$this->deletePastData();

				return $errors;
			}
		}


		// otp validation failed or no otp provided
		$errors->add('otp_error', __('Invalid OTP entered!', 'textlk-wp'));

		return $errors;
	}

	/**
	 * Validate guest checkout otp
	 *
	 * @param $errors
	 * @param $sanitized_user_login
	 * @param $user_email
	 *
	 * @return mixed
	 */
	public function validate_guest_checkout_otp()
	{

		$enable_guest_checkout = get_option('woocommerce_enable_guest_checkout');
		$enable_guest_checkout = $enable_guest_checkout === 'yes' ? true : false;

		if (!$this->pluginActive || !$this->options['otp_checkout'] || !$enable_guest_checkout) {
			return;
		}

		if (!empty($_REQUEST['otp_code'])) {
			$otp_code = sanitize_text_field($_REQUEST['otp_code']);

			$valid_user = $this->authenticate_otp(trim($otp_code));

			if ($valid_user) {
				$this->deletePastData();
			} else {
				wc_add_notice(__('Please enter a valid OTP.', 'textlk-wp'), 'error');
			}
		} else {
			wc_add_notice(__('Please enter a valid OTP.', 'textlk-wp'), 'error');
		}
	}


	/**
	 * Select otp from db and compare
	 *
	 * @param $otp_code
	 *
	 * @return bool
	 */
	public function authenticate_otp($otp_code)
	{


		if (!empty($_SESSION['textlk_otp_code']) && !empty($_SESSION['textlk_expires'])) {

			if (strtotime($_SESSION['textlk_expires']) > strtotime(TEXTLK_WP_TIMESTAMP)) {
				if ($otp_code === $_SESSION['textlk_otp_code']) {
					return true;
				}
			}
		}

		return false;
	}

	/**
	 * delete db data of current ip address user
	 *
	 */
	public function deletePastData()
	{
		if (isset($_SESSION['textlk_otp_code'], $_SESSION['textlk_expires'])) {
			unset($_SESSION['textlk_otp_code'], $_SESSION['textlk_expires']);
		}
	}

	/**
	 * Woocommerce validate phone and validate otp
	 *
	 * @param $errors
	 * @param $sanitized_user_login
	 * @param $user_email
	 *
	 * @return mixed
	 */
	public function wc_register_form_validation($errors, $sanitized_user_login, $user_email)
	{
		if (!$this->pluginActive) {
			return $errors;
		}

		if ($this->options['otp_checkout'] || ($this->options['wc_reg'] && $_POST['action_type'] === 'wc_reg')) {
			$this->register_form_validation($errors, $sanitized_user_login, $user_email);
		}

		return $errors;
	}

	/**
	 * Alert customer and admins when a new order is placed
	 *
	 * @param $order_id
	 */
	public function wc_new_order_alert($order_id)
	{
		if (!$order_id) {
			return;
		}

		// new order status pending notification for customer
		$this->wc_order_status_change_alert($order_id, 'pending', 'pending');

		// option not enabled
		if (!$this->pluginActive || !isset($this->options['order_status_admin']) || !$this->options['order_status_admin']) {
			return;
		}


		// send sms to all admins if enabled
		$order = new WC_Order($order_id);

		$admin_msg = $this->options['ADMIN_STATUS_SMS'];

		$search = [
			'[store_name]',
			'[billing_first_name]',
			'[order_id]',
			'[order_status]',
			'[order_currency]',
			'[order_amount]',
			'[order_date_created]',
			'[order_date_completed]'
		];

		$order_created = gmdate('d M Y', strtotime($order->get_date_created()));
		$order_completed = !empty($order->get_date_completed()) ? gmdate('d M Y', strtotime($order->get_date_completed())) : '';

		$replace = [
			get_bloginfo(),
			$order->get_billing_first_name(),
			$order_id,
			'pending',
			$order->get_currency(),
			$order->get_total(),
			$order_created,
			$order_completed
		];

		$admin_msg = str_replace($search, $replace, $admin_msg);

		// if admin phone is not provided then send to all admins
		$admin_phones[] = $this->options['admin_phones'];

		if (empty($admin_phones)) {
			$admin_phones = $this->admin_phones();
		}

		if (!empty($admin_phones)) {
			$numbers = implode(',', $admin_phones);
			$this->SendSMS($numbers, $admin_msg);
		}
	}

	/**
	 * Alert customer and user when order status changes
	 *
	 * @param $order_id
	 * @param $old_status
	 * @param $new_status
	 */

	public function wc_order_status_change_alert($order_id, $old_status, $new_status)
	{
		if (!$order_id) {
			return;
		}

		$order = new WC_Order($order_id);

		// Get the Customer billing phone
		$billing_phone = $order->get_billing_phone();

		//we will send sms
		$status = str_replace('-', '_', $order->data['status']);

		// option not enabled
		if (
			!$this->pluginActive || !isset($this->options['order_status_' . $status]) || !$this->options['order_status_' . $status]
		) {
			return;
		}

		$buyer_msg = !empty($this->options['ORDER_STATUS_' . strtoupper($status) . '_SMS']) ? $this->options['ORDER_STATUS_' . strtoupper($status) . '_SMS'] : NULL;

		$search = [
			'[store_name]',
			'[billing_first_name]',
			'[order_id]',
			'[order_status]',
			'[order_currency]',
			'[order_amount]',
			'[order_date_created]',
			'[order_date_completed]'
		];

		$order_created = gmdate('d M Y', strtotime($order->get_date_created()));
		$order_completed = !empty($order->get_date_completed()) ? gmdate('d M Y', strtotime($order->get_date_completed())) : '';

		$replace = [
			get_bloginfo(),
			$order->get_billing_first_name(),
			$order_id,
			$new_status,
			$order->get_currency(),
			$order->get_total(),
			$order_created,
			$order_completed
		];

		$buyer_msg = str_replace($search, $replace, $buyer_msg);

		if (empty($buyer_msg)) {
			$order->add_order_note( $this->plugin_name . ': ' . __( 'Order message not found.', 'textlk-wp' ) );
			return;
		}

		$response = $this->SendSMS($billing_phone, $buyer_msg);

		if ($response->error === 0) {

			$order->add_order_note(__('Text.lk SMS : Notified customer about his order ', 'textlk-wp') . $order->data['status'] . ' status');
		} else {
			$order->add_order_note(__('Text.lk SMS : ', 'textlk-wp') . $response->msg);
		}
	}

	/**
	 * Get all the phone number associated with administration role
	 *
	 * @return array
	 */
	public function admin_phones()
	{
		$admin_ids = get_users(['fields' => 'ID', 'role' => 'administrator']);
		$numbers   = [];
		foreach ($admin_ids as $userid) {
			$number = $this->validateNumber(get_user_meta($userid, 'mobile_phone', true));
			if ($number) {
				$numbers[] = $number;
			}
		}

		return $numbers;
	}

	/**
	 * WordPress login with Phone Number methods
	 *
	 */

	public function login_enqueue_style()
	{
		if ($this->options['wp_login'] || $this->options['wp_reg']) {
			wp_enqueue_style(
				$this->plugin_name,
				plugin_dir_url(__FILE__) . 'css/otp-login-form.css',
				[],
				$this->version,
				'all'
			);
		}
	}

	public function login_enqueue_script()
	{
		if (!$this->pluginActive) {
			return;
		}

		if ($this->options['wp_login'] || $this->options['wp_reg']) {
			wp_enqueue_script(
				$this->plugin_name,
				plugin_dir_url(__FILE__) . 'js/otp-login-form.js',
				['jquery'],
				$this->version,
				false
			);
			wp_localize_script(
				$this->plugin_name,
				$this->plugin_name . '_object',
				['ajaxurl' => admin_url('admin-ajax.php')]
			);
		}
	}

	/**
	 * Add OTP view in Wp login form
	 *
	 */
	public function add_otp_field_in_wp_login_form()
	{
		if (!$this->pluginActive || !$this->options['wp_login']) {
			return;
		}

		require_once 'partials/add-otp-on-login-form.php';
	?>
		<input type='hidden' name='action_type' id='action_type' value='wp_login' />
	<?php
	}

	/**
	 * Add OTP view in Wc login form
	 *
	 */
	public function add_otp_field_in_wc_login_form()
	{
		if (!$this->pluginActive || !$this->options['wc_login']) {
			return;
		}
		require_once 'partials/add-otp-on-login-form.php';
	?>
		<input type='hidden' name='action_type' id='action_type' value='wc_login' />
		<?php
	}

	/**
	 * Verify number and send otp
	 *
	 */
	public function save_and_send_otp_login() {
		check_ajax_referer('ajax-login-nonce', $this->plugin_name);

		$info = [];
		$info['user_login']    = isset($_POST['log']) ? sanitize_text_field($_POST['log']) : '';
		$info['user_password'] = isset($_POST['pwd']) ? $_POST['pwd'] : ''; // Don't over-sanitize passwords
		$info['remember']      = isset($_POST['rememberme']) ? sanitize_text_field($_POST['rememberme']) : '';

		// Attempt to get user by login or email
		$userdata = get_user_by('login', $info['user_login']);
		if (!$userdata) {
			$userdata = get_user_by('email', $info['user_login']);
		}

		if (!$userdata || !isset($userdata->data->ID, $userdata->data->user_pass)) {
			$response = ['status' => 401, 'message' => __('Wrong username or password!', 'textlk-wp')];
			echo wp_kses_post(json_encode($response));
			wp_die();
			exit;
		}

		$user_id = $userdata->data->ID;
		$hashed_password = $userdata->data->user_pass;

		// Validate and check password
		if (empty($info['user_password']) || !is_string($hashed_password) || !wp_check_password($info['user_password'], $hashed_password, $user_id)) {
			$response = ['status' => 401, 'message' => __('Wrong username or password!', 'textlk-wp')];
			echo wp_kses_post(json_encode($response));
			wp_die();
			exit;
		}

		// Try to get user phone number
		$user_phone = get_user_meta($user_id, 'mobile_phone', true);
		if (!$user_phone) {
			$user_phone = get_user_meta($user_id, 'billing_phone', true);
		}

		// Validate phone number
		if (!$user_phone || !$this->validateNumber($user_phone)) {
			$response = ['status' => 402, 'message' => __('No valid phone number found', 'textlk-wp')];
			echo wp_kses_post(json_encode($response));
			wp_die();
			exit;
		}

		// Generate OTP
		$otp_code = $this->generateOTP();
		$number   = $user_phone;
		$body     = 'Your OTP for ' . get_bloginfo() . ' login is: ' . $otp_code . '. Valid for 2 minutes.';

		$sms_response = $this->SendSMS($number, $body);

		if (is_object($sms_response) && $sms_response->error === 0) {
			$log_info = $this->log_login_register_action($user_phone, $otp_code);

			if ($log_info) {
				$response = ['status' => 200, 'message' => 'Please enter the verification code sent to your phone.'];
			} else {
				$response = ['status' => 500, 'message' => 'Something went wrong. Please try again.'];
			}

			echo wp_kses_post(json_encode($response));
			exit;
		}

		$response = ['status' => 400, 'message' => 'Error sending OTP code. Please contact administrator.'];
		echo wp_kses_post(json_encode($response));
		wp_die();
		exit;
	}

	/**
	 * Login the user verifying otp code
	 *
	 * @param $user
	 * @param $username
	 *
	 * @return User|WP_Error
	 */
	public function login_user($user, $username)
	{
		if (empty($user->data)) {
			return $user;
		}
		if (!$this->pluginActive || (!$this->options['wp_login'] && !$this->options['wc_login'])) {
			return $user;
		}

		if (empty($_POST['action_type'])) {
			$error = new WP_Error();

			$error->add(
				'empty_password',
				__('<strong>Error</strong>: Authentication Error!', 'textlk-wp')
			);
		}

		if (($this->options['wp_login'] && $_POST['action_type'] == 'wp_login') ||
			($this->options['wc_login'] && $_POST['action_type'] == 'wc_login')
		) {
			return $this->startOTPChallenge($user, $username);
		}

		return $user;
	}

	/**
	 * @param $user
	 * @param $username
	 *
	 * @return mixed|WP_Error
	 */
	public function startOTPChallenge($user, $username)
	{
		$user_phone = get_user_meta($user->data->ID, 'mobile_phone', true);

		if (!$user_phone) {
			$user_phone = get_user_meta($user->data->ID, 'billing_phone', true);
		}

		if (!$user_phone || !$this->validateNumber($user_phone)) {
			return $user;
		}

		if (empty($_REQUEST['otp_code'])) {
			$error = new WP_Error();

			$error->add(
				'empty_password',
				__('<strong>Error</strong>: Wrong OTP Code!', 'textlk-wp')
			);

			return $error;
		}

		$otp_code = sanitize_text_field($_REQUEST['otp_code']);

		$valid_user = $this->authenticate_otp($otp_code);

		if ($valid_user) {
			$this->deletePastData();

			return $user;
		}

		return new WP_Error(
			'invalid_password',
			__('OTP is not valid', 'textlk-wp')
		);
	}

	/**
	 * Woocommerce otp form in checkout
	 */
	public function otp_form_at_checkout()
	{
		if (!$this->pluginActive || !$this->options['otp_checkout']) {
			return;
		}

		if (!is_user_logged_in()) {
			require_once 'partials/add-otp-checkout-form.php';
		?>
			<input type='hidden' name='action_type' id='action_type' value='wc_checkout' />
<?php
		}
	}

	/**
	 * Check if entered api key is valid or not
	 *
	 * @return bool
	 */
	private function checkAPI($textlk_sms_api_key)
	{
		require_once TEXTLK_WP_PATH . 'includes/sms.class.php';

		$smsPortal = new TextLK_SMS($textlk_sms_api_key);

		$response = $smsPortal->getBalance();

		return $response && $response->error === 0;
	}
}
